/*----------------------------------------------------------------------------

 Copyright (C) Sartorius Stedim Data Analytics AB 2017 -

 Use, modification and distribution are subject to the Boost Software
 License, Version 1.0. (See http://www.boost.org/LICENSE_1_0.txt)
 ----------------------------------------------------------------------------
*/

#include "stdafx.h"
#include "sqmrunner.h"
#include <string.h>
#include <stdlib.h>
#include "utf8util.h"

static int CleanupWorkset(SQ_Workset hWorkset, FILE* pErr);
static int SaveModel(SQ_Workset hWorkset, FILE* pErr);

SQ_Project SQMRunner_Import(const char* szUSPName, FILE* pIn, FILE* pErr)
{
   SQ_Import hImportHandle = NULL;
   SQ_Project pProject = NULL;
   unsigned iPos;
   unsigned iRow = 1;
   unsigned iCol = 1;
   size_t count;
   int iMaxCount;
   float fVal;
   char* buffer;
   char* tmpBuffer;
   int bRetVal = 1;
   SQ_ErrorCode eSuccess = SQ_E_OK;
   SQ_StringMatrix oObservationIDs = NULL;
   SQ_FloatMatrix oData = NULL;
   SQ_StringMatrix oQualData = NULL;
   SQ_StringMatrix oVariableIDs = NULL;
   SQ_StringMatrix oQualVariableIDs = NULL;
   SQ_StringVector oszNames = NULL;
   SQ_StringVector oszObsNames = NULL;
   SQ_Bool bValid;
   const char* szErrString;
   unsigned iNumRows = 0;
   unsigned iNumCols = 0;
   unsigned iNumCurCols = 0;

   if (SQ_IsLicenseFileValid(&bValid) != SQ_E_OK)
   {
      szErrString = "Could not read license..";
      goto std_error_exit;
   }
   else if (!bValid)
   {
      szErrString = "Invalid license..";
      goto std_error_exit;
   }

   if (SQ_InitImport(szUSPName, NULL, &hImportHandle) != SQ_E_OK)
   {
      szErrString = "Could not create project. InitImport failed.";
      goto std_error_exit;
   }

   /*Read the input file.*/
   /*Decide the file size*/
   iPos = fseek(pIn, 0, SEEK_END);
   if (iPos)
      goto error_file_exit;

   iMaxCount = ftell(pIn);
   iPos = fseek(pIn, 0, SEEK_SET);
   if (iPos)
      goto error_file_exit;

   buffer = (char*)malloc(iMaxCount + 1);
   if (buffer == NULL)
      goto error_file_exit;

   count = fread(buffer, sizeof(char), iMaxCount, pIn);
   if (ferror(pIn) || count < 5)
      goto error_file_exit;

   buffer[count] = '\0';
   /*Decide the size of the data.*/
   iPos = 0;

   while (iPos < count)
   {
      if (buffer[iPos] == '\t')
         iNumCurCols++;

      if (buffer[iPos] == '\n')
      {
         iNumCurCols++;
         if (iNumCols > 0 && iNumCurCols != iNumCols)
            goto error_file_exit;

         iNumCols = iNumCurCols;
         iNumRows++;
         iNumCurCols = 0;
      }
      iPos++;
   }
   if (SQ_InitStringMatrix(&oObservationIDs, iNumRows - 2, 2) != SQ_E_OK ||
      SQ_InitFloatMatrix(&oData, iNumRows - 2, iNumCols - 4) != SQ_E_OK ||
      SQ_InitStringMatrix(&oQualData, iNumRows - 2, 2) != SQ_E_OK ||
      SQ_InitStringMatrix(&oVariableIDs, 2, iNumCols - 4) != SQ_E_OK ||
      SQ_InitStringMatrix(&oQualVariableIDs, 2, 2) != SQ_E_OK ||
      SQ_InitStringVector(&oszNames, 2) != SQ_E_OK ||
      SQ_InitStringVector(&oszObsNames, 2) != SQ_E_OK)
   {
      UTF8_printf(pErr, "%s\n", "Matrix Initialization failed");
      bRetVal = 0;
      goto exit1;
   }

   if (iNumCols < 4 || iNumRows < 4)
      goto error_file_exit;

   tmpBuffer = &buffer[0];

   for (iRow = 1; iRow <= iNumRows; iRow++)
   {
      for (iCol = 1; iCol <= iNumCols; iCol++)
      {
         iPos = 0;
         while (tmpBuffer[iPos] != '\t' && tmpBuffer[iPos] != '\n' && tmpBuffer[iPos] != '\0')
            iPos++;

         tmpBuffer[iPos] = '\0';

         if (iRow == 1 && iCol <= 2)
         {
            eSuccess = SQ_SetStringInVector(oszObsNames, iCol, tmpBuffer);
            if (iCol == 1 && eSuccess == SQ_E_OK)
            {
               eSuccess = SQ_SetStringInVector(oszNames, iRow, tmpBuffer);
            }
         }
         else if (iCol == 1 && iRow <= 2)
         {
            eSuccess = SQ_SetStringInVector(oszNames, iRow, tmpBuffer);
         }
         else if (iCol > 2 && iCol <= 4 && iRow <= 2)
         {
            eSuccess = SQ_SetStringInMatrix(oQualVariableIDs, iRow, iCol - 2, tmpBuffer);
         }
         else if (iCol > 4 && iRow <= 2)
         {
            eSuccess = SQ_SetStringInMatrix(oVariableIDs, iRow, iCol - 4, tmpBuffer);
         }
         else if (iCol <= 2 && iRow > 2)
         {
            eSuccess = SQ_SetStringInMatrix(oObservationIDs, iRow - 2, iCol, tmpBuffer);
         }
         else if (iCol <= 4 && iRow > 2)
         {
            eSuccess = SQ_SetStringInMatrix(oQualData, iRow - 2, iCol - 2, tmpBuffer);
         }
         else if (iRow > 2 && iCol > 4)
         {
            fVal = (float)atof(tmpBuffer);
            eSuccess = SQ_SetDataInFloatMatrix(oData, iRow - 2, iCol - 4, fVal);
         }

         if (eSuccess != SQ_E_OK)
         {
            UTF8_printf(pErr, "%s\n", "Matrix operation failed");
            bRetVal = 0;
            goto exit1;
         }
         tmpBuffer = &tmpBuffer[iPos + 1];
      }
   }
   free(buffer);
   if (
      /*Reserve space for the import, this will make the import faster if you call*/
      SQ_Reserve(hImportHandle, iNumRows - 2, iNumCols - 4, 2) != SQ_E_OK ||
      /*Set the observation names in the new dataset.*/
      SQ_SetObservationNames(hImportHandle, &oObservationIDs, 1) != SQ_E_OK ||
      /*Set a value the the import will treat as a missing value.*/
      SQ_SetMissingValueRepresentation(hImportHandle, -99) != SQ_E_OK ||
      /*Set a value the the import will treat as a missing value.*/
      SQ_SetMissingValueStringRepresentation(hImportHandle, "N/A") != SQ_E_OK ||
      /*Add variables to the new dataset.*/
      SQ_AddQuantitativeVariables(hImportHandle, &oData, &oVariableIDs, 1) != SQ_E_OK ||
      /*Add qualitative vars to variables to the new dataset.*/
      SQ_AddQualitativeVariables(hImportHandle, &oQualData, &oQualVariableIDs, 1) != SQ_E_OK ||
      /*Name the secondary IDs, the primary ID is always called PrimaryID an can't be renamed*/
      SQ_SetVariableIDSeriesNames(hImportHandle, &oszNames) != SQ_E_OK ||
      /*Name the secondary IDs, the primary ID is always called PrimaryID an can't be renamed*/
      SQ_SetObservationIDSeriesNames(hImportHandle, &oszObsNames) != SQ_E_OK ||
      /*Set the name of the new dataset*/
      SQ_SetDataSetName(hImportHandle, "DataSet1") != SQ_E_OK ||
      /*Create a dataset of the variables and observations added and release the import handle*/
      SQ_FinishImport(&hImportHandle, &pProject) != SQ_E_OK)
   {
      UTF8_printf(pErr, "%s\n", "Import function failed");
      bRetVal = 0;
      goto exit1;
   }

exit1:
   SQ_ClearStringMatrix(&oObservationIDs);
   SQ_ClearFloatMatrix(&oData);
   SQ_ClearStringMatrix(&oQualData);
   SQ_ClearStringMatrix(&oVariableIDs);
   SQ_ClearStringMatrix(&oQualVariableIDs);
   SQ_ClearStringVector(&oszNames);
   SQ_ClearStringVector(&oszObsNames);
   return pProject;
error_file_exit:
   UTF8_printf(pErr, "File Reading error");
   return NULL;

std_error_exit:
   UTF8_printf(pErr, "%s\n", szErrString);
   return NULL;
}

int SQMRunner_Workset(SQ_Project pProject, FILE* pErr)
{
   const char* szErrString = "";         /* String that will contain possible error. */
   SQ_Workset hWorkset = NULL;            /* The workset handle */
   SQ_IntVector vecDatasets = NULL;        /* Dataset numbers to build the workset from. */

   if (SQ_InitIntVector(&vecDatasets, 1) != SQ_E_OK)
   {
      UTF8_printf(pErr, "%s\n", "Vector initialization failed");
      goto error_exit;
   }

   if (SQ_SetDataInIntVector(vecDatasets, 1, 1) != SQ_E_OK)
   {
      UTF8_printf(pErr, "%s\n", "Vector operation failed");
      goto error_exit;
   }

   /* Get a new workset handle */
   if (SQ_GetNewWorkset(pProject, vecDatasets, &hWorkset) != SQ_E_OK)
   {
      szErrString = "SQ_GetNewWorkset failed.";
      goto error_exit;
   }

   /* Create a default workset. */
   if (SQ_CreateDefaultWorkset(hWorkset) != SQ_E_OK)
   {
      szErrString = "SQ_CreateDefaultWorkset failed.";
      goto error_exit;
   }

   /* Set name, description, missing value tolerance and save the model */
   if (!SaveModel(hWorkset, pErr))
      return 0;

   /* Release the workset and close the project */
   if (!CleanupWorkset(hWorkset, pErr))
      return 0;
   return 1;

error_exit:
   UTF8_printf(pErr, "%s", szErrString);
   return 0;
}

int SaveModel(SQ_Workset hWorkset, FILE* pErr)
{
   char* szErrString;
   SQ_IntVector oModelNumbers = NULL;
   SQ_Bool bCreateClassModels = SQ_False;
   SQ_ModelType eModelType = SQ_PCA_X;

   if (SQ_SetWorksetDescription(hWorkset, "This model was created by SQMCSample.exe") != SQ_E_OK)
   {
      szErrString = "SQ_SetWorksetDescription failed.";
      goto error_exit;
   }

   /* Create a new model for the workset. */
   if (SQ_CreateModel(hWorkset, bCreateClassModels, eModelType, &oModelNumbers) != SQ_E_OK)
   {
      szErrString = "SQ_CreateModel failed.";
      goto error_exit;
   }
   SQ_ClearIntVector(&oModelNumbers);

   return 1;
error_exit:
   UTF8_printf(pErr, "%s", szErrString);
   return 0;
}

int CleanupWorkset(SQ_Workset hWorkset, FILE* pErr)
{
   const char* szErrString;           /* String that will contain possible error. */

   /* Release the handle */
   if (SQ_ReleaseWorkset(hWorkset) != SQ_E_OK)
   {
      szErrString = "SQ_ReleaseWorkset failed.";
      goto error_exit;
   }

   return 1;

error_exit:
   UTF8_printf(pErr, "%s", szErrString);
   return 0;
}

int SQMRunner_Fit(SQ_Project pProject, FILE* pErr)
{
   const char* szErrString;
   int iNumComponents;
   int iComp;
   SQ_Model pModel = NULL;

   if (SQ_GetModel(pProject, 1, &pModel) != SQ_E_OK)
   {
      szErrString = "SQMRunner_Fit(): SQ_GetModel failed.";
      goto error_msg;
   }

   /* Auto fit the first model  */
   if (SQ_AutofitModel(pModel) != SQ_E_OK)
   {
      szErrString = "SQMRunner_Fit(): SQ_AutofitModel failed.";
      goto error_msg;
   }

   if (SQ_GetNumberOfComponents(pModel, &iNumComponents) == SQ_E_OK)
   {
      /* Make sure that the model has at least two components after this sample has been run. */
      for (iComp = iNumComponents; iComp < 2; ++iComp)
      {
         if (SQ_CalculateNextComponent(pModel) != SQ_E_OK)
         {
            szErrString = "SQMRunner_Fit(): SQ_CalculateNextComponent failed.";
            goto error_msg;
         }
      }
   }

   return 1;

error_msg:
   UTF8_printf(pErr, "%s\n", szErrString);
   return 0;
}

